/*******************************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.corext.refactoring.reorg;

import org.eclipse.core.resources.IResource;

import org.eclipse.ltk.core.refactoring.participants.ReorgExecutionLog;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;

/**
 * Logged implementation of new name queries.
 *
 * @since 3.3
 */
public final class LoggedNewNameQueries implements INewNameQueries {

	/** Default implementation of a new name query */
	private final class NewNameQuery implements INewNameQuery {

		/** The name */
		private final String fName;

		/** The object */
		private final Object fObject;

		/**
		 * Creates a new new name query.
		 *
		 * @param object
		 *            the object
		 * @param name
		 *            the initial suggested name
		 */
		public NewNameQuery(final Object object, String name) {
			fObject= object;
			fName= name;
		}

		/**
		 * Returns the new name of the compilation unit, without any extension.
		 *
		 * @return the new name, or <code>null</code>
		 */
		private String getCompilationUnitName() {
			String name= fLog.getNewName(fObject);
			if (name != null) {
				int index= name.lastIndexOf('.');
				if (index > 0)
					name= name.substring(0, index);
			}
			return name;
		}

		@Override
		public String getNewName() {
			String name= null;
			if (fObject instanceof ICompilationUnit)
				name= getCompilationUnitName();
			else
				name= fLog.getNewName(fObject);
			if (name == null)
				name= fName;
			return fName;
		}
	}

	/** The reorg execution log */
	private final ReorgExecutionLog fLog;

	/**
	 * Creates a new logged new name queries.
	 *
	 * @param log
	 *            the reorg execution log
	 */
	public LoggedNewNameQueries(final ReorgExecutionLog log) {
		fLog= log;
	}

	@Override
	public INewNameQuery createNewCompilationUnitNameQuery(final ICompilationUnit unit, final String initialSuggestedName) {
		return new NewNameQuery(unit, initialSuggestedName);
	}

	@Override
	public INewNameQuery createNewPackageFragmentRootNameQuery(final IPackageFragmentRoot root, final String initialSuggestedName) {
		return new NewNameQuery(root, initialSuggestedName);
	}

	@Override
	public INewNameQuery createNewPackageNameQuery(final IPackageFragment fragment, final String initialSuggestedName) {
		return new NewNameQuery(fragment, initialSuggestedName);
	}

	@Override
	public INewNameQuery createNewResourceNameQuery(final IResource resource, final String initialSuggestedName) {
		return new NewNameQuery(resource, initialSuggestedName);
	}

	@Override
	public INewNameQuery createNullQuery() {
		return createStaticQuery(null);
	}

	@Override
	public INewNameQuery createStaticQuery(final String name) {
		return new INewNameQuery() {

			@Override
			public String getNewName() {
				return name;
			}
		};
	}
}
